-- AnimalFoodCalculator LS25
-- Anzeige des geschätzten Futterverbrauchs (Tag / Jahr) pro Stall im Tiermenü

-- Idee inspiriert von LS22 AnimalFoodCalculator
-- LS25 Version eigenständig neu implementiert

-- Autor: [WeekendFarmers] T4xs
-- Version 1.0.0.0 initial FS25 release 16.09.2025


AnimalFoodCalculator = {}
AnimalFoodCalculator.name = g_currentModName or "AnimalFoodCalculator"
AnimalFoodCalculator.debug = false
AnimalFoodCalculator.data = {} -- Verbrauchswerte pro Stall (keyed by placeable ID)
AnimalFoodCalculator.daysPerPeriod = nil


-- =====================================================
-- Helper: Logging
-- =====================================================
local function afcInfo(fmt, ...)
    if AnimalFoodCalculator.debug then
        Logging.info("[AFC] " .. string.format(fmt, ...))
    end
end

local function afcWarn(fmt, ...)
    Logging.warning("[AFC] " .. string.format(fmt, ...))
end

-- =====================================================
-- Helper: Cluster-Ermittlung
-- =====================================================
local function getClustersFromObject(obj)
    if obj == nil then return nil end

    if type(obj.getClusters) == "function" then
        local ok, res = pcall(function() return obj:getClusters() end)
        if ok and type(res) == "table" then return res end
    end

    if type(obj.clusters) == "table" then
        return obj.clusters
    end

    if type(obj.spec_husbandryAnimals) == "table" then
        local spec = obj.spec_husbandryAnimals
        if type(spec.getClusters) == "function" then
            local ok, res = pcall(function() return spec:getClusters() end)
            if ok and type(res) == "table" then return res end
        elseif type(spec.clusters) == "table" then
            return spec.clusters
        end
    end

    if obj.owningPlaceable and type(obj.owningPlaceable) == "table" then
        local place = obj.owningPlaceable
        if place.spec_husbandryAnimals and type(place.spec_husbandryAnimals) == "table" then
            local spec = place.spec_husbandryAnimals
            if type(spec.getClusters) == "function" then
                local ok, res = pcall(function() return spec:getClusters() end)
                if ok and type(res) == "table" then return res end
            elseif type(spec.clusters) == "table" then
                return spec.clusters
            end
        end
    end

    if type(obj.getModuleByName) == "function" then
        local ok, mod = pcall(function() return obj:getModuleByName("husbandryAnimals") end)
        if ok and type(mod) == "table" then
            if type(mod.getClusters) == "function" then
                local ok2, res2 = pcall(function() return mod:getClusters() end)
                if ok2 and type(res2) == "table" then return res2 end
            elseif type(mod.clusters) == "table" then
                return mod.clusters
            end
        end
    end

    return nil
end

-- =====================================================
-- Helper: Stall-Key bestimmen
-- =====================================================
local function getHusbandryKey(husbandry)
    if not husbandry then return nil end
    if husbandry.owningPlaceable and husbandry.owningPlaceable.id then
        return string.format("placeable_%d", husbandry.owningPlaceable.id)
    end
    if husbandry.owningPlaceable and husbandry.owningPlaceable.name then
        return "placeable_name_" .. tostring(husbandry.owningPlaceable.name)
    end
    return tostring(husbandry)
end

-- EAS-Erkennung (EnhancedAnimalSystem)
AnimalFoodCalculator.isEAS = g_modManager and g_modManager:getModByName("FS25_EnhancedAnimalSystem") ~= nil


-- =====================================================
-- Berechnung: Verbrauch
-- =====================================================
-- Berechnung: Food/age helpers (dynamisch mit Map/Mods + EAS-Laktationsfaktor)
function AnimalFoodCalculator:calcFoodByAge(cluster, extraMonths)
    if not cluster or cluster.subTypeIndex == nil then 
        return 0 
    end

    local subTypes = g_currentMission and g_currentMission.animalSystem and g_currentMission.animalSystem.subTypes
    if not subTypes then 
        return 0 
    end

    local subType = subTypes[cluster.subTypeIndex]
    if not subType or not subType.input or not subType.input["food"] then 
        return 0 
    end

    -- Alter + extra Monate
    local idx = (cluster.age or 0) + (extraMonths or 0)
    if idx < 0 then idx = 0 end

    local foodInput = subType.input["food"]
    local baseVal = 0
    if type(foodInput.get) == "function" then
        baseVal = foodInput:get(idx)
    elseif type(foodInput[idx]) == "number" then
        baseVal = foodInput[idx]
    end

    -- Patch: EAS-Faktor, zusätzlich zum Map-/Mod-Wert
    if AnimalFoodCalculator.isEAS then
    local species = string.lower(subType.name or "")
    if (species == "cow" or species == "goat") and cluster.lactating then
        local lactationFactor = cluster.lactationFactor or 1
        baseVal = baseVal * lactationFactor
    end
end


    return baseVal
end



function AnimalFoodCalculator:calcFoodPerCluster(cluster)
    if not cluster or cluster.numAnimals == nil then return 0, 0 end
    local num = cluster.numAnimals
    local first = self:calcFoodByAge(cluster, 0)
    local whole = first
    for m = 1, 11 do
        whole = whole + self:calcFoodByAge(cluster, m)
    end
    -- Giants-Default (1 Tag = 1 Monat).  Fallback nur, wenn wirklich nichts gesetzt ist.
    local daysPerMonth = 1

    if g_currentMission ~= nil then
        local env = g_currentMission.environment
        if env ~= nil and env.daysPerPeriod ~= nil and env.daysPerPeriod > 0 then
            daysPerMonth = env.daysPerPeriod
        elseif g_currentMission.missionInfo ~= nil and g_currentMission.missionInfo.daysPerPeriod ~= nil and g_currentMission.missionInfo.daysPerPeriod > 0 then
            daysPerMonth = g_currentMission.missionInfo.daysPerPeriod
        end
    end

    local daily = (first * num) / daysPerMonth
    local yearly = whole * num
    return daily, yearly
end

-- =====================================================
-- Event: Tiere geändert
-- =====================================================
function AnimalFoodCalculator:onAnimalsChanged(husbandryAnimals)
    if not husbandryAnimals then return end
    local clusters = getClustersFromObject(husbandryAnimals)
    if not clusters then return end

    local sd, sy = 0, 0
    for _, c in ipairs(clusters) do
        local ok, d, y = pcall(function() return self:calcFoodPerCluster(c) end)
        if ok then sd = sd + (d or 0); sy = sy + (y or 0) end
    end

    local key = getHusbandryKey(husbandryAnimals)
    if key then
        self.data[key] = { daily = sd, yearly = sy }
        husbandryAnimals.afcDaily = sd
        husbandryAnimals.afcYearly = sy
    end

    afcInfo("onAnimalsChanged: stored key=%s daily=%.2f yearly=%.2f", tostring(key), sd, sy)
end

-- =====================================================
-- GUI Hook
-- =====================================================
local function AFC_hook_updateHusbandryDisplay(self, husbandry, ...)
    if not husbandry then return end

    AnimalFoodCalculator:onAnimalsChanged(husbandry)

    local key = getHusbandryKey(husbandry)
    local daily, yearly = nil, nil

    if key and AnimalFoodCalculator.data[key] then
        daily = AnimalFoodCalculator.data[key].daily
        yearly = AnimalFoodCalculator.data[key].yearly
    else
        -- fallback: on-the-fly berechnen (keine Speicherung)
        local clusters = getClustersFromObject(husbandry)
        if clusters then
            local sd, sy = 0, 0
            for _, c in ipairs(clusters) do
                local ok, d, y = pcall(function() return AnimalFoodCalculator:calcFoodPerCluster(c) end)
                if ok then sd = sd + (d or 0); sy = sy + (y or 0) end
            end
            daily, yearly = sd, sy
            -- store for later (if key exists)
            if key then AnimalFoodCalculator.data[key] = { daily = daily, yearly = yearly } end
        end
    end

    if daily == nil then return end

    -- sichere Kandidatenliste: wir versuchen zuerst left-title-like elements (keine Value-Spalte)
    local titleCandidates = { "foodRowTotalTitle", "foodRowTitle", "foodRowTitleText", "foodRowTotal" }
    local valueCandidates = { "foodRowTotalValue", "foodRowValue", "foodRowValueText", "capacityText", "foodCapacityElement" }

    local titleEl, valueEl = nil, nil
    for _, k in ipairs(titleCandidates) do
        if self[k] and type(self[k]) == "table" and type(self[k].getText) == "function" and type(self[k].setText) == "function" then
            titleEl = self[k]
            break
        end
    end
    for _, k in ipairs(valueCandidates) do
        if self[k] and type(self[k]) == "table" and type(self[k].getText) == "function" and type(self[k].setText) == "function" then
            valueEl = self[k]
            break
        end
    end

    -- if we didn't find a dedicated title element, fallback to a generic candidate list used before
    if not titleEl then
        local genericCandidates = { "foodRowTotalValue", "foodRowTotal", "foodRowValue", "foodRowTotalTitle" }
        for _, k in ipairs(genericCandidates) do
            if self[k] and type(self[k]) == "table" and type(self[k].getText) == "function" and type(self[k].setText) == "function" then
                titleEl = self[k]
                break
            end
        end
    end

    if not titleEl then
        afcInfo("GUI-Hook: kein Title-Element gefunden für Anzeige")
        return
    end

    -- base text must be stored per husbandry key to avoid cross-over
    local baseText = nil
    if key then
        baseText = (husbandry._afcBaseText and husbandry._afcBaseText[key]) or nil
    end
    if not baseText then
        baseText = titleEl:getText() or ""
        if key then
            husbandry._afcBaseText = husbandry._afcBaseText or {}
            husbandry._afcBaseText[key] = baseText
        else
            titleEl._afcBaseText = titleEl._afcBaseText or baseText
        end
    end

    -- i18n format fallback
    local fmt = "Verbrauch/Tag: %.0f l, Jahr: %.0f l"
    if g_i18n and type(g_i18n.getText) == "function" then
        local ok, s = pcall(function() return g_i18n:getText("afc_gui_format") end)
        if ok and type(s) == "string" and s ~= "" then fmt = s end
    end

    local extra = string.format(fmt, daily, yearly)

    -- ----- Sichere Anzeige: prefix an die Value-Spalte (ohne globales Überschreiben) -----
    local function safeTrim(s)
        if type(s) ~= "string" then return s end
        return s:gsub("^%s*(.-)%s*$", "%1")
    end

    -- versuchen wir zuerst die vorhandene Value-Spalte zu erweitern (bevor wir klonen)
    if valueEl and type(valueEl.getText) == "function" and type(valueEl.setText) == "function" then
        local ok, cur = pcall(function() return valueEl:getText() or "" end)
        if not ok then cur = "" end

        -- falls der Text bereits unseren Prefix enthält, extrahiere das "echte" Basis-Right-Teil
        local base = cur
        local sepPos = cur:find("|")
        if sepPos then
            base = cur:sub(sepPos + 1)
            base = safeTrim(base)
        end

        -- setze neuen zusammengesetzten Text: Verbrauch | originalValue
        local composed = string.format("%s   |   %s", extra, (base ~= "" and base) or cur)
        pcall(function() valueEl:setText(composed) end)

    else
        -- Fallback: wenn keine Value-Spalte vorhanden, sichere Clone-Option (ohne Positionierung)
        if self.afcConsumptionText == nil then
            local refEl = titleEl or valueEl
            if refEl and type(refEl.clone) == "function" and refEl.parent and refEl.parent.addElement then
                local ok, clone = pcall(function() return refEl:clone(self) end)
                if ok and clone and type(clone.setText) == "function" then
                    clone:setVisible(true)
                    clone:setText(extra)
                    refEl.parent:addElement(clone) -- Giants-Layout setzt Position
                    self.afcConsumptionText = clone
                end
            end
        else
            pcall(function() self.afcConsumptionText:setText(extra) end)
        end
    end
end

-- =====================================================
-- Registrierung
-- =====================================================
Mission00.loadMission00Finished = Utils.appendedFunction(Mission00.loadMission00Finished, function()
        -- Server oder SP: initialisiere daysPerPeriod
    if g_server ~= nil or g_dedicatedServerInfo ~= nil then
        if g_currentMission and g_currentMission.missionInfo and g_currentMission.missionInfo.daysPerPeriod then
            AnimalFoodCalculator.daysPerPeriod = g_currentMission.missionInfo.daysPerPeriod
        end
    end

    g_messageCenter:subscribe(MessageType.HUSBANDRY_ANIMALS_CHANGED,
        function(h) AnimalFoodCalculator:onAnimalsChanged(h) end)

    if InGameMenuAnimalsFrame ~= nil then
        local orig = InGameMenuAnimalsFrame.updateHusbandryDisplay
        if type(orig) == "function" then
            InGameMenuAnimalsFrame.updateHusbandryDisplay =
                Utils.appendedFunction(orig, AFC_hook_updateHusbandryDisplay)
            afcInfo("GUI-Hook registriert")
        else
            afcInfo("updateHusbandryDisplay nicht vorhanden - GUI-Hook übersprungen")
        end
    else
        afcInfo("InGameMenuAnimalsFrame nicht gefunden - GUI-Hook übersprungen")
    end
end)

-- ENDE
